#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion

namespace gov.va.med.vbecs.BOL.UnitTests
{

	
	

	[TestFixture]
	public class BOL_ExceptionReport : BaseTest 
	{
		//TODO: Should really have a UnitTest exception type
		Common.ExceptionType _exceptionType = Common.ExceptionType.ABO_incompatible_unit_selected;
		string _exceptionComment = System.DateTime.Now.ToString();

		BOL.ExceptionReport _testExReport;
		BOL.Patient _testPatient;
		BOL.Specimen _testSpecimen;
		BOL.BloodUnit _testBloodUnit;

		private DataTable _dtIssuedUnit;
		private const string _INPUT_STR = "Wahoo";
	

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{
				this._dtIssuedUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 IU.* FROM IssuedUnit IU JOIN vw_BloodShipAllCurrent VW ON IU.BloodUnitGuid = VW.BloodUnitGuid AND VW.DivisionCode = '" + this.TestDivisionCode + "' AND (VW.UnitStatusCode = 'I' OR VW.UnitStatusCode = 'I')  WHERE IU.ReturnDateTime IS NULL ORDER BY IU.LastUpdateDate DESC",  true);

				this._testBloodUnit = new BOL.BloodUnit(this.GetRecordGuid(TABLES.BloodUnit.TableName, TABLES.BloodUnit.BloodUnitGuid,true));
				this._testPatient = new BOL.Patient(this.GetRecordGuid(TABLES.Patient.TableName, TABLES.Patient.PatientGuid,false));
				this._testSpecimen = new BOL.Specimen(this.GetRecordGuid(TABLES.PatientSpecimen.TableName, TABLES.PatientSpecimen.PatientSpecimenGuid,true));

				this.RefreshData = false;
			}

			this._testExReport = new BOL.ExceptionReport(this._exceptionType, this._exceptionComment, Common.LogonUser.LogonUserName);
		}

		[Test]
		public void Ctor_Blank_Pass()
		{
			BOL.ExceptionReport tmpER = new BOL.ExceptionReport();
			//
			Assert.IsNotNull(tmpER);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Blank_Fail()
		{
		}

		[Test]
		public void Ctor_Pass()
		{
			BOL.ExceptionReport testExc = new BOL.ExceptionReport(Common.ExceptionType.ABO_incompatible_unit_selected, _INPUT_STR, Common.LogonUser.LogonUserName);

			Assert.IsTrue(testExc.ExceptionType == Common.ExceptionType.ABO_incompatible_unit_selected);
			Assert.IsTrue(testExc.ExceptionComment == _INPUT_STR);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Fail()
		{
		}

		[Test]
		public void ExceptionType_Pass()
		{
			Common.ExceptionType typTest = Common.ExceptionType.Discrepant_patient_antigen_typing;

			this._testExReport.ExceptionType = typTest;

			Assert.IsTrue(_testExReport.ExceptionType == typTest);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ExceptionType_Fail()
		{
		}

		[Test]
		public void ExceptionData_Pass()
		{
			DataRow drTest = new DataTable().NewRow();

			this._testExReport.ExceptionData = drTest;

			Assert.IsTrue(this._testExReport.ExceptionData == drTest);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ExceptionData_Fail()
		{
		}

		[Test]
		public void ExceptionComment_Pass()
		{
			string strTest = System.DateTime.Now.ToString();

			this._testExReport.ExceptionComment = strTest;

			Assert.IsTrue(this._testExReport.ExceptionComment == strTest);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ExceptionComment_Fail()
		{
		}

		[Test]
		public void ExceptionDate_Pass()
		{
			System.DateTime dTest = System.DateTime.Now;
			
			this._testExReport.ExceptionDate = dTest;

			Assert.IsTrue(this._testExReport.ExceptionDate == dTest);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ExceptionDate_Fail()
		{
		}

		[Test]
		public void ExceptionUser_Pass()
		{
			string strTest = System.DateTime.Now.ToString();
			
			this._testExReport.ExceptionUser = strTest;

			Assert.IsTrue(this._testExReport.ExceptionUser == strTest);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ExceptionUser_Fail()
		{
		}

		[Test]
		public void GenerateExceptionQC_Pass()
		{
			
			UnitTestUtil.RunSQL("UPDATE RACK SET RecordStatusCode = 'A' WHERE RackGuid = (SELECT TOP 1 RackGuid FROM Rack WHERE DailyControlGuid IS NOT NULL AND RecordStatusCode = 'X' AND DivisionCode = '" + this.TestDivisionCode + "')");
			DataTable dtDailyQC = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 DailyQcExpirationDate FROM DailyQcWorklist WHERE DivisionCode = '" + this.TestDivisionCode + "' AND RecordStatusCode = 'A'");
			this._testExReport.ExceptionDate = (System.DateTime)dtDailyQC.Rows[0][TABLES.DailyQcWorklist.DailyQcExpirationDate];
			this._testExReport.ExceptionDate = System.DateTime.UtcNow;

			System.Guid rackGuid = DataCreator.GetRecordGuid("SELECT TOP 1 RackGuid FROM Rack WHERE DailyControlGuid IS NOT NULL AND RecordStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "'");			
			//Gotta update the DailyQcWorklist for Rack constructor
			UnitTestUtil.RunSQL("UPDATE DailyQcWorklist SET DailyQcExpirationDate = GETUTCDATE(), RecordStatusCode = 'A' WHERE RackGuid = '" + rackGuid.ToString() + "'");
			System.Guid reagentGuid = this.GetRecordGuid(TABLES.Reagent.TableName, TABLES.Reagent.ReagentGuid, true);

			this._testExReport.GenerateExceptionQC(reagentGuid, rackGuid, Guid.Empty, Guid.Empty, "0","1",ARTIFICIAL.IS, Common.LogonUser.LogonUserName);
 
			Assert.AreEqual(18,this._testExReport.ExceptionData.Table.Columns.Count, "Columns.Count");
			Assert.AreEqual("0", this._testExReport.ExceptionData[TABLES.ExceptionQc.PreviousResult].ToString(), "PreviousResult");
			Assert.AreEqual("1", this._testExReport.ExceptionData[TABLES.ExceptionQc.CurrentResult].ToString(), "Current Result");
		}

		[Test]
		public void GenerateExceptionQC_Fail()
		{		
			this._testExReport.GenerateExceptionQC(System.Guid.NewGuid(), System.Guid.NewGuid(), Guid.Empty, Guid.Empty, "0", "1", ARTIFICIAL.IS, Common.LogonUser.LogonUserName);

			Assert.IsNull(this._testExReport.ExceptionData);
		}

		[Test]
		public void GenerateExceptionQCInactivation_Pass()
		{
			DataTable dtTemp = new DataTable();
			dtTemp.Columns.Add(TABLES.ExceptionQcInactivationReagent.ReagentTypeName, typeof(string));
			dtTemp.Columns.Add(TABLES.ExceptionQcInactivationReagent.TestWithName, typeof(string));
			dtTemp.Columns.Add("IsResults", typeof(string));
			dtTemp.Columns.Add("IatResults", typeof(string));
			dtTemp.Columns.Add("CcResults", typeof(string));

			DataRow dr = dtTemp.NewRow();
			dr[TABLES.ExceptionQcInactivationReagent.ReagentTypeName] = _INPUT_STR;
			dr[TABLES.ExceptionQcInactivationReagent.TestWithName] = _INPUT_STR;
			dr["IsResults"] = "0";
			dr["IatResults"] = "1";
			dr["CcResults"] = "2";
			dtTemp.Rows.Add(dr);

			DataTable dtTest = this._testExReport.GenerateExceptionQCInactivation(_INPUT_STR, dtTemp, "testingtech");

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionQcInactivation.ReagentRack].ToString() == _INPUT_STR);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 8);
			Assert.IsTrue(dtTest.Rows[0][TABLES.ExceptionQcInactivationReagent.ReagentTypeName].ToString() == _INPUT_STR);
			Assert.IsTrue(dtTest.Rows[0][TABLES.ExceptionQcInactivationReagent.TestWithName].ToString() == _INPUT_STR);
			Assert.IsTrue(dtTest.Rows[0][TABLES.ExceptionQcInactivationReagent.IsResult].ToString() == "0");
			Assert.IsTrue(dtTest.Rows[0][TABLES.ExceptionQcInactivationReagent.IatResult].ToString() == "1");
			Assert.IsTrue(dtTest.Rows[0][TABLES.ExceptionQcInactivationReagent.CcResult].ToString() == "2");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionQCInactivation_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionQCNotPerformed_Pass()
		{
			this._testExReport.GenerateExceptionQCNotPerformed(this._testPatient, this._testSpecimen, _INPUT_STR, Common.LogonUser.LogonUserName);
			
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 12);
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionQcNotPerformed.RackName].ToString() == _INPUT_STR);
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionQcNotPerformed.PatientGuid].ToString() == this._testPatient.PatientGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionQcNotPerformed.SpecimenUid].ToString() == this._testSpecimen.SpecimenUid);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionQCNotPerformed_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionQCNotPerformed_4Strings_Pass()
		{
			this._testExReport.GenerateExceptionQCNotPerformed(this._testBloodUnit.EyeReadableUnitId,this._testBloodUnit.ProductCode,"A",Environment.UserName);

			Assert.IsNotNull(this._testExReport.ExceptionData);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 10);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionQCNotPerformed_4Strings_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionPatientAboRhDiscrepancy_Pass()
		{
			this._testExReport.GenerateExceptionPatientAboRhDiscrepancy(this._testPatient, "12345", "A", "P", Common.LogonUser.LogonUserName, Common.LogonUser.LogonUserName, Common.LogonUser.LogonUserDivisionCode, System.DateTime.Now.AddYears(-1), "98765", "B", "N", Common.LogonUser.LogonUserName, Common.LogonUser.LogonUserName, Common.LogonUser.LogonUserDivisionCode, System.DateTime.Now);

			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 21);
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionPatientAboRhDiscrepancy.PatientGuid].ToString() == this._testPatient.PatientGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionPatientAboRhDiscrepancy.PreviousSpecimenUid].ToString() == "12345");
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionPatientAboRhDiscrepancy.CurrentSpecimenUid].ToString() == "98765");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionPatientAboRhDiscrepancy_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionReflexTest_Pass()
		{
			System.Guid patientSpecimenGuid = this.GetRecordGuid(TABLES.PatientSpecimen.TableName, TABLES.PatientSpecimen.PatientSpecimenGuid,true);
			BOL.Specimen tstSpecimen = new BOL.Specimen(patientSpecimenGuid);

			this._testExReport.GenerateExceptionReflexTest(tstSpecimen, _INPUT_STR);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionReflexTest.TestOrdered].ToString() == _INPUT_STR);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 13);
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionReflexTest.SpecimenUid].ToString() == tstSpecimen.SpecimenUid);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionReflexTest_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionSpecimen_Pass()
		{
			Guid TRGuid = Guid.NewGuid();
			Guid patientGuid = this._testSpecimen.Patient.PatientGuid;
			
			string insertTransfusionReq = "INSERT INTO PATIENTTRANSFUSIONREQUIREMENT (";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementText + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDivisionCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.VistaConversionIndicator + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.AntibodyTypeId + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.RecordStatusCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateFunctionId + ")";
			insertTransfusionReq += "VALUES (";
			insertTransfusionReq += "'" + TRGuid.ToString() + "',";
			insertTransfusionReq += "'" + patientGuid.ToString() + "',";
			insertTransfusionReq += "'Anti-C',";
			insertTransfusionReq += "'AS',";
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertTransfusionReq += "'1',";
			insertTransfusionReq += "'5',";
			insertTransfusionReq += "'C',";  //TR from conversion
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertTransfusionReq);
			
			this._testExReport.GenerateExceptionSpecimen(null, this._testSpecimen);

			string deleteTRs = "DELETE FROM PATIENTTRANSFUSIONREQUIREMENT WHERE PatientTransfusionRequirementGuid = '" + TRGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteTRs);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionSpecimen.SpecimenUid].ToString() == this._testSpecimen.SpecimenUid);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 18);
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionSpecimen.PatientAntibodies].ToString().IndexOf("Anti-C") != -1);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionSpecimen_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionPatientAntigenDiscrepancy_Pass()
		{
			System.Guid oTestGuid = this.GetRecordGuid(TABLES.OrderedTest.TableName, TABLES.OrderedTest.OrderedTestGuid, true);
			DataTable dtTest = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 PO.PatientOrderGuid FROM PatientOrder PO JOIN PatientTreatment PT 	ON PO.PatientTreatmentGUID = PT.PatientTreatmentGUID JOIN PatientSpecimen PS	ON PT.PatientGUID = PS.PatientGUID JOIN OrderedTest OT 		ON PS.PatientSpecimenGUID = OT.PatientSpecimenGUID WHERE PO.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);
			System.Guid pOrderGuid = (System.Guid)dtTest.Rows[0][TABLES.PatientOrder.PatientOrderGuid];
			
			BOL.OrderedTest tstOrder = new BOL.OrderedTest(oTestGuid, pOrderGuid);

			this._testExReport.GenerateExceptionPatientAntigenDiscrepancy(tstOrder, _INPUT_STR, _INPUT_STR, _INPUT_STR, _INPUT_STR, System.DateTime.Now, "N", "P", Common.LogonUser.LogonUserName);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionPatientAntigenDiscrepancy.ReagentName].ToString() == _INPUT_STR);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 18);
		}

		[Test]
		public void GenerateExceptionPatientAntigenDiscrepancy_Fail()
		{		
			this._testExReport.GenerateExceptionPatientAntigenDiscrepancy(null, _INPUT_STR, _INPUT_STR, _INPUT_STR, _INPUT_STR, DateTime.Now, _INPUT_STR, _INPUT_STR, _INPUT_STR);

			Assert.IsNull(this._testExReport.ExceptionData);
		}

		[Test]
		public void GenerateExceptionPatientTestInactivated_Pass()
		{
			System.Guid oTestGuid = this.GetRecordGuid(TABLES.OrderedTest.TableName, TABLES.OrderedTest.OrderedTestGuid, true);
			DataTable dtTest = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 PO.PatientOrderGuid FROM PatientOrder PO JOIN PatientTreatment PT 	ON PO.PatientTreatmentGUID = PT.PatientTreatmentGUID JOIN PatientSpecimen PS	ON PT.PatientGUID = PS.PatientGUID JOIN OrderedTest OT 		ON PS.PatientSpecimenGUID = OT.PatientSpecimenGUID WHERE PO.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);
			System.Guid pOrderGuid = (System.Guid)dtTest.Rows[0][TABLES.PatientOrder.PatientOrderGuid];
			
			BOL.OrderedTest tstOrder = new BOL.OrderedTest(oTestGuid, pOrderGuid);

			this._testExReport.GenerateExceptionPatientTestInactivated(tstOrder, Common.LogonUser.LogonUserName);

			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 14);
		}

		[Test]
		public void GenerateExceptionPatientTestInactivated_Fail()
		{		
			this._testExReport.GenerateExceptionPatientTestInactivated(null, Common.LogonUser.LogonUserName);

			Assert.IsNull(this._testExReport.ExceptionData);
		}

		[Test]
		public void GenerateExceptionPatientABORh_Pass()
		{
			this._testExReport.GenerateExceptionPatientABORh(this._testPatient, "12345", "A", "N", "A", "N", Common.LogonUser.LogonUserName, DateTime.Now);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionPatientAboRh.SpecimenUid].ToString() == "12345");
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 15);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionPatientABORh_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionPatientTest_Pass()
		{
			System.Guid oTestGuid = this.GetRecordGuid(TABLES.OrderedTest.TableName, TABLES.OrderedTest.OrderedTestGuid, true);
			DataTable dtTest = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 PO.PatientOrderGuid FROM PatientOrder PO JOIN PatientTreatment PT 	ON PO.PatientTreatmentGUID = PT.PatientTreatmentGUID JOIN PatientSpecimen PS	ON PT.PatientGUID = PS.PatientGUID JOIN OrderedTest OT 		ON PS.PatientSpecimenGUID = OT.PatientSpecimenGUID WHERE PO.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);
			System.Guid pOrderGuid = (System.Guid)dtTest.Rows[0][TABLES.PatientOrder.PatientOrderGuid];
			
			BOL.OrderedTest tstOrder = new BOL.OrderedTest(oTestGuid, pOrderGuid);

			this._testExReport.GenerateExceptionPatientTest(tstOrder, ARTIFICIAL.IS, "P", DateTime.Now, Common.LogonUser.LogonUserName);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionPatientTest.ReactivityPhase].ToString() == ARTIFICIAL.IS);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 15);
		}

		[Test]
		public void GenerateExceptionPatientTest_Fail()
		{		
			this._testExReport.GenerateExceptionPatientTest(null, ARTIFICIAL.IS, "P", DateTime.MinValue, Common.LogonUser.LogonUserName);

			Assert.IsNull(this._testExReport.ExceptionData);
		}

		[Test]
		public void GenerateExceptionQuarantine_Pass()
		{
			this._testExReport.GenerateExceptionQuarantine(this._testBloodUnit);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionQuarantine.UnitId].ToString() == this._testBloodUnit.EyeReadableUnitId);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 17);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionQuarantine_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionUnitAboRh_Pass()
		{
			this._testExReport.GenerateExceptionUnitAboRh(this._testBloodUnit.BloodUnitGuid, "A", "P", "Test User1");

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionUnitAboRh.UnitGuid].ToString() == this._testBloodUnit.BloodUnitGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 13);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionUnitAboRh_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionUnitModification_Pass()
		{

			DataRow tmpData = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM Supply WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");

			BOL.Supply testSupply=new BOL.Supply(tmpData);

			BOL.BloodUnitModification tstBum = new BOL.BloodUnitModification(Common.ProductModification.Washed, Common.ModifyUnitMethod.Open, System.DateTime.Now, Common.LogonUser.LogonUserName);

			//this._testExReport.GenerateExceptionUnitModification(this._testBloodUnit, "Wash", Common.LogonUser.LogonUserName, System.DateTime.Now, System.DateTime.Now);
			this._testExReport.GenerateExceptionUnitModification(this._testBloodUnit, tstBum, System.DateTime.Now.AddDays(5), testSupply);
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionUnitModification.UnitGuid].ToString() == this._testBloodUnit.BloodUnitGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 21);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionUnitModification_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionUnitSelection_Pass()
		{
			Guid TRGuid = Guid.NewGuid();
			Guid patientGuid = this._testPatient.PatientGuid;
			
			string insertTransfusionReq = "INSERT INTO PATIENTTRANSFUSIONREQUIREMENT (";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementText + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDivisionCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.VistaConversionIndicator + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.AntibodyTypeId + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.RecordStatusCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateFunctionId + ")";
			insertTransfusionReq += "VALUES (";
			insertTransfusionReq += "'" + TRGuid.ToString() + "',";
			insertTransfusionReq += "'" + patientGuid.ToString() + "',";
			insertTransfusionReq += "'Anti-C',";
			insertTransfusionReq += "'AS',";
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertTransfusionReq += "'1',";
			insertTransfusionReq += "'5',";
			insertTransfusionReq += "'C',";  //TR from conversion
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertTransfusionReq);

			this._testExReport.GenerateExceptionUnitSelection(this._testBloodUnit.BloodUnitGuid, this._testPatient, System.DateTime.Now, Common.LogonUser.LogonUserName);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionUnitSelection.UnitGuid].ToString() == this._testBloodUnit.BloodUnitGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 23);
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionUnitSelection.PatientAntibodies].ToString().IndexOf("Anti-C") != -1);

			string deleteTRs = "DELETE FROM PATIENTTRANSFUSIONREQUIREMENT WHERE PatientTransfusionRequirementGuid = '" + TRGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteTRs);

		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionUnitSelection_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionLabelMismatch_Pass()
		{
			this._testExReport.GenerateExceptionLabelMismatch(this._testBloodUnit, "Wash", Common.LogonUser.LogonUserName, DateTime.Now, "123","456");

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionLabelMismatch.TargetUnitGuid].ToString() == this._testBloodUnit.BloodUnitGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 17);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionLabelMismatch_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionExpiredUnitReceived_Pass()
		{
			this._testExReport.GenerateExceptionExpiredUnitReceived(this._testBloodUnit);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionExpiredUnitReceived.UnitGuid].ToString() == this._testBloodUnit.BloodUnitGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 15);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionExpiredUnitReceived_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionReturnIssuedUnit_Pass()
		{
			System.Guid issuedUnitGuid = this.GetRecordGuid(TABLES.IssuedUnit.TableName, TABLES.IssuedUnit.IssuedUnitGuid, false);

			BOL.IssuedUnit tstUnit = new BOL.IssuedUnit(issuedUnitGuid);
			this._testExReport.GenerateExceptionReturnIssuedUnit(tstUnit, this._testPatient, this._testBloodUnit);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionReturnedUnit.PatientGuid].ToString() == this._testPatient.PatientGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionReturnedUnit.ReturnedUnitGuid].ToString() == this._testBloodUnit.BloodUnitGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionReturnedUnit.IssueToLocation].ToString() == tstUnit.IssueToLocationName);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 18);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionReturnIssuedUnit_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionAboIncompatibleTransfusion_Pass()
		{
			this._testExReport.GenerateExceptionAboIncompatibleTransfusion(this._testPatient, this._testBloodUnit);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionTransfusion.TransfusedPatientGuid].ToString() == this._testPatient.PatientGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionTransfusion.UnitGuid].ToString() == this._testBloodUnit.BloodUnitGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 15);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionAboIncompatibleTransfusion_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionTrwUnitClericalCheckFailed_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtIssuedUnit.Rows[0][TABLES.IssuedUnit.BloodUnitGuid];

			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid);

			System.Guid orderedTestGuid = this.GetRecordGuid(TABLES.OrderedTest.TableName, TABLES.OrderedTest.OrderedTestGuid, true);

			BOL.PatientTransfusionReaction tstPTR = new BOL.PatientTransfusionReaction(orderedTestGuid);
			BOL.TransfusionReaction tstTR = new BOL.TransfusionReaction();

			this._testExReport.GenerateExceptionTrwUnitClericalCheckFailed(tstPTR, this._testPatient, tmpUnit, tstTR);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionTransfusionReactionWorkup.AssignedPatientGuid].ToString() == this._testPatient.PatientGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionTransfusionReactionWorkup.UnitGuid].ToString() == tmpUnit.BloodUnitGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 24);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionTrwUnitClericalCheckFailed_Fail()
		{		
		}

		[Test]
		public void GenerateExceptionTrwSpecimen_Pass()
		{
			System.Guid orderedTestGuid = this.GetRecordGuid(TABLES.OrderedTest.TableName, TABLES.OrderedTest.OrderedTestGuid, true);

			BOL.PatientTransfusionReaction tstPTR = new BOL.PatientTransfusionReaction(orderedTestGuid);

			this._testExReport.GenerateExceptionTrwSpecimen(tstPTR, this._testPatient, this._testSpecimen, true);

			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionTrwSpecimen.TrwPatientGuid].ToString() == this._testPatient.PatientGuid.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionTrwSpecimen.SpecimenUid].ToString() == this._testSpecimen.SpecimenUid);
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionTrwSpecimen.TrwDate].ToString() == tstPTR.InvestigatedDateTime.ToString());
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 16);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateExceptionTrwSpecimen_Fail()
		{		
		}

		[Test]
		public void GetExceptionReport_Pass()
		{		
			int counter = 0;
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);

			foreach(Common.ExceptionReportType type in Enum.GetValues(typeof(Common.ExceptionReportType)))
			{
				if(type!=Common.ExceptionReportType.Unknown)
				{
					DataSet dsExceptionReport = BOL.ExceptionReport.GetExceptionReport( startDate, endDate, type );
					counter += dsExceptionReport.Tables.Count;
				}
			}			
			Assert.IsTrue(counter > 0);
		}

		[Test]
		public void GetExceptionReport_Fail()
		{		
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);
			Common.ExceptionReportType exType = Common.ExceptionReportType.ExceptionReportMain;

			DataSet dsTest = BOL.ExceptionReport.GetExceptionReport(startDate, endDate, exType);
			Assert.IsTrue(dsTest.Tables.Count == 0);
		}

		[Test]
		public void GetPatientExceptionReport_Pass()
		{		
			System.Guid patientGuid = this._testPatient.PatientGuid;
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);
			Common.ExceptionReportType exType;
			DataSet dsTest;

			//GetExceptionReportPatientAboRh
			exType = Common.ExceptionReportType.ExceptionReportPatientAboRh;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportPatientAboRH(dsTest);

			//GetExceptionReportPatientAboRHDiscrepancy
			exType = Common.ExceptionReportType.ExceptionReportPatientAboRhDiscrepancy;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportPatientAboRHDiscrepancy(dsTest);

			//GetExceptionReportPatientAntigenDiscrepancy
			exType = Common.ExceptionReportType.ExceptionReportPatientAntigenDiscrepancy;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportPatientAntigenDiscrepancy(dsTest);

			//GetExceptionReportPatientTest
			exType = Common.ExceptionReportType.ExceptionReportPatientTest;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportPatientTest(dsTest);

			//GetExceptionReportPatientTestInactivated
			exType = Common.ExceptionReportType.ExceptionReportPatientTestInactivated;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportPatientTestInactivated(dsTest);

			//GetExceptionReportQCNotPerformed
			exType = Common.ExceptionReportType.ExceptionReportQCNotPerformed;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportQCNotPerformed(dsTest, true);

			//GetExceptionReportReflexTest
			exType = Common.ExceptionReportType.ExceptionReportReflexTest;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportReflexTest(dsTest);

			//GetExceptionReportSpecimen
			exType = Common.ExceptionReportType.ExceptionReportSpecimen;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportSpecimen(dsTest);

			//GetExceptionReportUnitSelection
			exType = Common.ExceptionReportType.ExceptionReportUnitSelection;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportUnitSelection(dsTest);

			//GetExceptionReportReturnIssuedUnit
			exType = Common.ExceptionReportType.ExceptionReportReturnIssuedUnit;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportReturnIssuedUnit(dsTest);

			//GetExceptionReportAboIncompatibleTransfusion
			exType = Common.ExceptionReportType.ExceptionReportAboIncompatibleTransfusion;
			dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportAboIncompatibleTransfusion(dsTest);

		}

		[Test]
		public void GetPatientExceptionReport_Fail()
		{		
			System.Guid patientGuid = this._testPatient.PatientGuid;
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);
			Common.ExceptionReportType exType = Common.ExceptionReportType.ExceptionReportMain;

			DataSet dsTest = BOL.ExceptionReport.GetPatientExceptionReport(patientGuid, exType);
			Assert.IsTrue(dsTest.Tables.Count == 0);
		}

		[Test]
		public void GetBloodUnitExceptionReport_Pass()
		{		
			System.Guid unitGuid = this._testBloodUnit.BloodUnitGuid;
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);
			Common.ExceptionReportType exType;
			DataSet dsTest;

			//GetExceptionReportLabelMismatch
			exType = Common.ExceptionReportType.ExceptionReportLabelMismatch;
			dsTest = BOL.ExceptionReport.GetBloodUnitExceptionReport(unitGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportLabelMismatch(dsTest);

			//GetExceptionReportQuarantine
			exType = Common.ExceptionReportType.ExceptionReportQuarantine;
			dsTest = BOL.ExceptionReport.GetBloodUnitExceptionReport(unitGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportQuarantine(dsTest);

			//GetExceptionReportUnitAboRH
			exType = Common.ExceptionReportType.ExceptionReportUnitAboRh;
			dsTest = BOL.ExceptionReport.GetBloodUnitExceptionReport(unitGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportUnitAboRh(dsTest);

			//GetExceptionReportUnitModification
			exType = Common.ExceptionReportType.ExceptionReportUnitModification;
			dsTest = BOL.ExceptionReport.GetBloodUnitExceptionReport(unitGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportUnitModification(dsTest);

			//GetExceptionReportUnitSelection
			exType = Common.ExceptionReportType.ExceptionReportUnitSelection;
			dsTest = BOL.ExceptionReport.GetBloodUnitExceptionReport(unitGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportUnitSelection(dsTest);

			//GetExceptionReportReturnIssuedUnit
			exType = Common.ExceptionReportType.ExceptionReportReturnIssuedUnit;
			dsTest = BOL.ExceptionReport.GetBloodUnitExceptionReport(unitGuid, exType);

			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportReturnIssuedUnit(dsTest);

//			//GetExceptionReportAboIncompatibleTransfusion
//			exType = Common.ExceptionReportType.ExceptionReportAboIncompatibleTransfusion;
//			dsTest = BOL.ExceptionReport.GetBloodUnitExceptionReport(unitGuid, exType);
//
//			DAL.UnitTests.DAL_ExceptionReport.ValidateExceptionReportAboIncompatibleTransfusion(dsTest);
		}

		[Test]
		public void GetBloodUnitExceptionReport_Fail()
		{		
			System.Guid unitGuid = this._testBloodUnit.BloodUnitGuid;
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);
			Common.ExceptionReportType exType = Common.ExceptionReportType.ExceptionReportMain;

			DataSet dsTest = BOL.ExceptionReport.GetBloodUnitExceptionReport(unitGuid, exType);
			Assert.IsTrue(dsTest.Tables.Count == 0);
		}

		[Test]
		public void GenerateExceptionPatientRestriction_Pass()
		{
			_testExReport.GenerateExceptionPatientRestriction(_testBloodUnit);
			//
			Assert.IsNotNull(this._testExReport.ExceptionData);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 17);
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void GenerateExceptionPatientRestriction_Fail()
		{
			_testExReport.GenerateExceptionPatientRestriction(null);
			//
			Assert.IsNull(this._testExReport.ExceptionData);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 17);
		}

		[Test]
		public void GetExceptionReportMain_Pass()
		{		
			DataSet dsTest = BOL.ExceptionReport.GetExceptionReportMain();

			Assert.IsTrue(dsTest.Tables.Count == 1);
			Assert.IsTrue(dsTest.Tables[0].TableName == "ExceptionReportMain");

			Assert.AreEqual(32,dsTest.Tables[0].Columns.Count);

			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionLabelMismatch.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionPatientAboRh.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionPatientAboRhDiscrepancy.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionPatientAntigenDiscrepancy.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionPatientTest.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionPatientTestInactivated.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionQc.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionQcInactivation.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionQcNotPerformed.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionQuarantine.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionReflexTest.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionSpecimen.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionUnitAboRh.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionUnitModification.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionUnitSelection.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionExpiredUnitReceived.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionTrwSpecimen.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionTransfusionReactionWorkup.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionAntigenTest.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains("ExceptionUnitTestSuppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionReturnedUnit.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionTransfusion.TableName + "Suppress"));
			Assert.IsTrue(dsTest.Tables[0].Columns.Contains(TABLES.ExceptionElectronicTransfusion.TableName + "Suppress"));
		}

		[Test]
		public void GenerateExceptionCautionTagBTRF_Pass()
		{
			_testExReport.GenerateExceptionCautionTagBTRF(_testPatient, _testSpecimen, _testBloodUnit);
			Assert.IsTrue( (Guid) _testExReport.ExceptionData[TABLES.ExceptionCautionTag.PatientGuid] == _testPatient.PatientGuid);
			Assert.IsTrue( (Guid) _testExReport.ExceptionData[TABLES.ExceptionCautionTag.UnitGuid] == _testBloodUnit.BloodUnitGuid);
		}
		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GenerateExceptionCautionTagBTRF_Patient_Fail()
		{
			_testExReport.GenerateExceptionCautionTagBTRF(null, _testSpecimen, _testBloodUnit);
		}
		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GenerateExceptionCautionTagBTRF_Unit_Fail()
		{
			_testExReport.GenerateExceptionCautionTagBTRF(_testPatient, _testSpecimen, null);
		}

		[Test]
		public void GenerateExceptionUnitIssue_Pass()
		{
			Guid TRGuid = Guid.NewGuid();
			Guid patientGuid = this._testPatient.PatientGuid;
			
			string insertTransfusionReq = "INSERT INTO PATIENTTRANSFUSIONREQUIREMENT (";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementText + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDivisionCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.VistaConversionIndicator + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.AntibodyTypeId + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.RecordStatusCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateFunctionId + ")";
			insertTransfusionReq += "VALUES (";
			insertTransfusionReq += "'" + TRGuid.ToString() + "',";
			insertTransfusionReq += "'" + patientGuid.ToString() + "',";
			insertTransfusionReq += "'Anti-C',";
			insertTransfusionReq += "'AS',";
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertTransfusionReq += "'1',";
			insertTransfusionReq += "'5',";
			insertTransfusionReq += "'C',";  //TR from conversion
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertTransfusionReq);
			
			_testExReport.GenerateExceptionUnitIssue(new BOL.PatientUnitIssue(_testPatient,new BOL.VbecsUser(Environment.UserName,false)),this._testBloodUnit);
			//
			Assert.IsNotNull(this._testExReport.ExceptionData);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 31);
			Assert.IsTrue(this._testExReport.ExceptionData[TABLES.ExceptionUnitIssue.PatientAntibodies].ToString().IndexOf("Anti-C") != -1);
		
			string deleteTRs = "DELETE FROM PATIENTTRANSFUSIONREQUIREMENT WHERE PatientTransfusionRequirementGuid = '" + TRGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteTRs);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GenerateExceptionUnitIssue_Fail()
		{
			_testExReport.GenerateExceptionUnitIssue(null, null);
			//
			Assert.IsNull(this._testExReport.ExceptionData);
		}

		[Test]
		public void GenerateExceptionEmergencyIssueUnitTest_Pass()
		{
			System.Guid patientGuid = this.GetRecordGuid(TABLES.Patient.TableName, TABLES.Patient.PatientGuid, false);
			System.Guid bloodUnitGuid = this.GetRecordGuid(TABLES.BloodUnit.TableName, TABLES.BloodUnit.BloodUnitGuid, true);

			this._testExReport.GenerateExceptionEmergencyIssueUnitTest(bloodUnitGuid, patientGuid, "UNIT TEST", "UNIT TEST", Environment.UserName);

			Assert.IsNotNull(this._testExReport.ExceptionData);
			Assert.IsTrue(this._testExReport.ExceptionData.Table.Columns.Count == 18);
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void GenerateExceptionEmergencyIssueUnitTest_Fail()
		{
			this._testExReport.GenerateExceptionEmergencyIssueUnitTest(Guid.Empty, Guid.Empty, null, null, null);

			Assert.IsNull(this._testExReport.ExceptionData);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetExceptionReportMain_Fail()
		{		
		}

		/// <summary>
		/// Gets the RecordGuid of the table
		/// </summary>
		/// <param name="tableName"></param>
		/// <param name="fieldName"></param>
		/// <param name="includeDivision"></param>
		/// <returns></returns>
		private System.Guid GetRecordGuid(string tableName, string fieldName, bool includeDivision)
		{
			string sqlText = "SELECT TOP 1 " + fieldName + " FROM " + tableName;

			if (includeDivision)
				sqlText += " WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'";

			System.Guid unitGuid = DataCreator.GetRecordGuid(sqlText);

			return(unitGuid);
		}

		[Test]
		public void GenerateExceptionEmergencyIssueOrder_Pass()
		{
			StringBuilder query = new StringBuilder( "SELECT TOP 1 * FROM OrderedComponent " );
			query.Append( "JOIN PatientOrder ON OrderedComponent.PatientOrderGuid = PatientOrder.PatientOrderGuid " );
			query.Append( "JOIN PatientTreatment ON PatientOrder.PatientTreatmentGuid = PatientTreatment.PatientTreatmentGuid " );
			query.Append( "JOIN Patient ON PatientTreatment.PatientGuid = Patient.PatientGuid " );
			query.Append( "WHERE PatientOrder.OrderReceivedDatetime IS NOT NULL" );
			//
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( query.ToString() );
			//
			BOL.OrderedComponent oc = new BOL.OrderedComponent( dr );
			//	
			this._testExReport.GenerateExceptionEmergencyIssueOrder( oc );
			//
			Assert.IsNotNull( this._testExReport.ExceptionData );
			Assert.IsTrue( this._testExReport.ExceptionData.Table.Columns.Count == 16 );
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GenerateExceptionEmergencyIssueOrder_Fail()
		{
			this._testExReport.GenerateExceptionEmergencyIssueOrder( null );

			Assert.IsNull( this._testExReport.ExceptionData );
		}

		[Test]
		public void GenerateExceptionRemoveFinalUnitStatus_Pass()
		{
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( "SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent" );
			//
			BOL.BloodUnit bu = new BOL.BloodUnit( ( Guid )dr[TABLES.BloodUnit.BloodUnitGuid] );
			//	
			this._testExReport.GenerateExceptionRemoveFinalUnitStatus( bu, Common.UnitStatusCode.Available );
			//
			Assert.IsNotNull( this._testExReport.ExceptionData );
			Assert.IsTrue( this._testExReport.ExceptionData.Table.Columns.Count == 17 );
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void GenerateExceptionRemoveFinalUnitStatus_Fail()
		{
			this._testExReport.GenerateExceptionRemoveFinalUnitStatus( null, Common.UnitStatusCode.Unknown );

			Assert.IsNull( this._testExReport.ExceptionData );
		}

		[Test]
		public void GenerateExceptionRemoveUnitModification_Pass()
		{
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( "SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent JOIN BloodUnitModification ON FromBloodUnitGuid = vw_BloodShipAllCurrent.BloodUnitGuid " );
			//
			Guid bloodUnitGuid = ( Guid )dr[TABLES.BloodUnit.BloodUnitGuid];
			//
			BOL.BloodUnit bu = new BOL.BloodUnit( bloodUnitGuid );
			//	
			dr = UnitTestUtil.RunSQLGetDataRow( string.Concat( "SELECT TOP 1 * FROM BloodUnitModification WHERE FromBloodUnitGuid = '", bloodUnitGuid, "'" ) );
			//
			this._testExReport.GenerateExceptionRemoveUnitModification( bu, dr );
			//
			Assert.IsNotNull( this._testExReport.ExceptionData );
			Assert.IsTrue( this._testExReport.ExceptionData.Table.Columns.Count == 17 );
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void GenerateExceptionRemoveUnitModification_Fail()
		{
			this._testExReport.GenerateExceptionRemoveUnitModification( null, null );

			Assert.IsNull( this._testExReport.ExceptionData );
		}

		[Test]
		public void GenerateExceptionAntigenTest_Pass()
		{
			StringBuilder query = new StringBuilder( "SELECT TOP 1 * FROM vw_BloodShipAllCurrent " );
			query.Append( "JOIN BloodUnitTest ON vw_BloodShipAllCurrent.BloodUnitGuid = BloodUnitTest.BloodUnitGuid " );
			query.Append( "JOIN TestResult ON BloodUnitTest.TestResultId = TestResult.TestResultId " );
			query.Append( "JOIN BloodTestType ON BloodUnitTest.BloodTestTypeId = BloodTestType.BloodTestTypeId " );
			query.Append( "JOIN OrderableTest ON BloodTestType.OrderableTestId = OrderableTest.OrderableTestId " );
			//
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( query.ToString() );
			//
			this._testExReport.GenerateExceptionAntigenTest( dr, "Test, Unit", BOL.VBECSDateTime.GetDivisionCurrentDateTime() );
			//
			Assert.IsTrue( this._testExReport.ExceptionData != null );
			Assert.IsTrue( this._testExReport.ExceptionData.Table.Columns.Count == 14 );
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void GenerateExceptionAntigenTest_Fail()
		{
			this._testExReport.GenerateExceptionAntigenTest( null, null, BOL.VBECSDateTime.GetDivisionCurrentDateTime() );

			Assert.IsNull( this._testExReport.ExceptionData );
		}

		[Test]
		public void GenerateExceptionUnitTest_Pass()
		{
			StringBuilder query = new StringBuilder( "SELECT TOP 1 BloodUnitGuid, EyeReadableUnitId FROM vw_BloodShipAllCurrent " );
			//
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( query.ToString() );
			//
			Guid bloodUnitGuid = (Guid)dr[TABLES.BloodUnit.BloodUnitGuid];
			string unitId = dr[TABLES.BloodUnit.EyeReadableUnitId].ToString();
			//
			this._testExReport.GenerateExceptionUnitTest( bloodUnitGuid, unitId, "", "", "", BOL.VBECSDateTime.GetDivisionCurrentDateTime(), "Test, Unit" );
			//
			Assert.IsTrue( this._testExReport.ExceptionData.Table.Columns.Count == 17 );
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void GenerateExceptionUnitTest_Fail()
		{
			this._testExReport.GenerateExceptionUnitTest( Guid.Empty, null, null, null, null, BOL.VBECSDateTime.GetDivisionCurrentDateTime(), null );

			Assert.IsNull( this._testExReport.ExceptionData );
		}

		[Test]
		public void GenerateExceptionExpiredTask_Pass()
		{
			StringBuilder query = new StringBuilder( "SELECT TOP 1 * FROM vw_BloodShipAllCurrent AS BloodUnit " );
			query.Append( "JOIN OrderedUnit ON BloodUnit.BloodUnitGuid = OrderedUnit.BloodUnitGuid " );
			query.Append( "JOIN OrderedComponent ON OrderedUnit.OrderedComponentGuid = OrderedComponent.OrderedComponentGuid " );
			query.Append( "JOIN PatientOrder ON OrderedComponent.PatientOrderGuid = PatientOrder.PatientOrderGuid " );
			query.Append( "JOIN PatientTreatment ON PatientOrder.PatientTreatmentGuid = PatientTreatment.PatientTreatmentGuid " );
			query.Append( "JOIN Patient ON PatientTreatment.PatientGuid = Patient.PatientGuid " );
			query.Append( "WHERE PatientOrder.OrderReceivedDatetime IS NOT NULL" );
			//
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( query.ToString() );
			BOL.OrderedComponent oc = new OrderedComponent( dr );
			//
			query.Remove( 0, query.Length );
			query.Append( "SELECT TOP 1 * FROM vw_BloodShipAllCurrent AS BloodUnit " );
			query.Append( "JOIN OrderedUnit ON BloodUnit.BloodUnitGuid = OrderedUnit.BloodUnitGuid " );
			query.Append( "JOIN OrderedTest ON OrderedUnit.OrderedTestGuid = OrderedTest.OrderedTestGuid " );
			query.Append( "JOIN PatientOrder ON OrderedTest.PatientOrderGuid = PatientOrder.PatientOrderGuid " );
			query.Append( "JOIN PatientTreatment ON PatientOrder.PatientTreatmentGuid = PatientTreatment.PatientTreatmentGuid " );
			query.Append( "JOIN Patient ON PatientTreatment.PatientGuid = Patient.PatientGuid " );
			query.Append( "WHERE PatientOrder.OrderReceivedDatetime IS NOT NULL" );
			//
			dr = UnitTestUtil.RunSQLGetDataRow( query.ToString() );
			BOL.OrderedTest ot = new OrderedTest( dr );
			//
			this._testExReport.GenerateExceptionExpiredTask( ot, oc, "Test, Unit" );
			//
			Assert.IsNotNull( this._testExReport.ExceptionData );
			Assert.IsTrue( this._testExReport.ExceptionData.Table.Columns.Count == 14 );
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void GenerateExceptionExpiredTask_Fail()
		{
			this._testExReport.GenerateExceptionExpiredTask( null, null, null );

			Assert.IsNull( this._testExReport.ExceptionData );
		}

		[Test]
		public void GenerateExceptionInconclusiveCrossmatch_Pass()
		{
			StringBuilder query = new StringBuilder( "SELECT TOP 1 *, OrderedComponent.PatientSpecimenGuid FROM vw_BloodShipAllCurrent AS BloodUnit " );
			query.Append( "JOIN OrderedUnit ON BloodUnit.BloodUnitGuid = OrderedUnit.BloodUnitGuid " );
			query.Append( "JOIN OrderedComponent ON OrderedUnit.OrderedComponentGuid = OrderedComponent.OrderedComponentGuid " );
			query.Append( "JOIN PatientOrder ON OrderedComponent.PatientOrderGuid = PatientOrder.PatientOrderGuid " );
			query.Append( "JOIN PatientTreatment ON PatientOrder.PatientTreatmentGuid = PatientTreatment.PatientTreatmentGuid " );
			query.Append( "JOIN Patient ON PatientTreatment.PatientGuid = Patient.PatientGuid " );
			query.Append( "WHERE PatientOrder.OrderReceivedDatetime IS NOT NULL AND OrderedComponent.PatientSpecimenGuid IS NOT NULL" );
			//
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( query.ToString() );
			Guid bloodUnitGuid = (Guid)dr[TABLES.BloodUnit.BloodUnitGuid]; 
			BOL.OrderedComponent oc = new OrderedComponent( dr );
			//
			this._testExReport.GenerateExceptionInconclusiveCrossmatch( oc, bloodUnitGuid, "", "", "Test, Unit" );
			//
			Assert.IsNotNull( this._testExReport.ExceptionData );
			Assert.IsTrue( this._testExReport.ExceptionData.Table.Columns.Count == 16 );
		}

		[Test]
		public void GenerateExceptionInconclusiveCrossmatch_Fail()
		{
			this._testExReport.GenerateExceptionInconclusiveCrossmatch( null, Guid.Empty, null, null, null );

			Assert.IsNull( this._testExReport.ExceptionData );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void GenerateAndPrintReport_Pass() 
		{
			// It's not possible to test printing from unit test.
		}

		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void GenerateAndPrintReport_Fail() 
		{
			// It's not possible to test printing from unit test.
		}

		// *****************************************************************

	}
}

#endif
